<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types = 1);

namespace Magento\FunctionalTestingFramework\Console;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;

class CleanProjectCommand extends Command
{
    const CONFIGURATION_FILES = [
        // codeception.yml file for top level config
        TESTS_BP . DIRECTORY_SEPARATOR . 'codeception.yml',
        // functional.suite.yml for test execution config
        TESTS_BP . DIRECTORY_SEPARATOR . 'tests' . DIRECTORY_SEPARATOR . 'functional.suite.yml',
        // Acceptance Tester Actions generated by codeception
        FW_BP . '/src/Magento/FunctionalTestingFramework/_generated',
        // AcceptanceTester Class generated by codeception
        FW_BP . '/src/Magento/FunctionalTestingFramework/AcceptanceTester.php'
    ];

    const GENERATED_FILES = [
        TESTS_MODULE_PATH . '/_generated'
    ];

    /**
     * Configures the current command.
     *
     * @return void
     */
    protected function configure()
    {
        $this->setName('reset')
            ->setDescription(
                'This command will clean any configuration files from the environment (not including .env), as well as generated artifacts.' // phpcs:ignore
            )
            ->addOption('hard', null, InputOption::VALUE_NONE, "parameter to force reset of configuration files.");
    }

    /**
     * Executes the current command.
     *
     * @param InputInterface  $input
     * @param OutputInterface $output
     * @return void
     * @throws \Symfony\Component\Console\Exception\LogicException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $isHardReset = $input->getOption('hard');
        $fileSystem = new Filesystem();
        $finder = new Finder();
        $finder->files()->name('*.php')->in(realpath(FW_BP . '/src/Magento/FunctionalTestingFramework/Group/'));
        $filesForRemoval = [];

        // include config files if user specifies a hard reset for deletion
        if ($isHardReset) {
            $filesForRemoval = array_merge($filesForRemoval, self::CONFIGURATION_FILES);
        }

        // include the files mftf generates during test execution in TESTS_BP for deletion
        $filesForRemoval = array_merge($filesForRemoval, self::GENERATED_FILES);

        if ($output->isVerbose()) {
            $output->writeln('Deleting Files:');
        }

        // delete any suite based group files
        foreach ($finder->files() as $file) {
            if ($output->isVerbose()) {
                $output->writeln($file->getRealPath());
            }

            $fileSystem->remove($file);
        }

        // delete files specified for removal
        foreach ($filesForRemoval as $fileForRemoval) {
            if ($fileSystem->exists($fileForRemoval) && $output->isVerbose()) {
                $output->writeln($fileForRemoval);
            }

            $fileSystem->remove($fileForRemoval);
        }

        $output->writeln('mftf files removed from filesystem.');
    }
}
